<?php

namespace App\Http\Controllers\Backend\Presentation;

use App\Http\Controllers\Controller;
use App\Http\Requests\Presentation\StorePresentationRequest;
use App\Http\Requests\Presentation\UpdatePresentationRequest;
use App\Models\Presentation;
use App\Traits\ImageUpload;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Yajra\DataTables\DataTables;

class PresentationController extends Controller
{
    use ImageUpload;
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $presentations = Presentation::select('id','title','details','details_bn','is_active', 'image')
            ->get();
        return view('backend.presentations.index', compact('presentations'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('backend.presentations.form');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(StorePresentationRequest $request)
    {
        DB::beginTransaction();
        try {
            $id = Presentation::create([
                'title' => $request->title,
                'title_bn' => $request->title_bn,
                'details' => $request->details,
                'details_bn' => $request->details_bn,
                'is_active' => $request->is_active ? 1 : 0,
            ])->id;
            if($request->hasFile('image')) {
                $imageName = $this->uploadImage('images/presentations/'.$id.'/', $request->image, Str::slug($request->title));
                Presentation::where('id', $id)->update(['image'=> $imageName]);
            }
            DB::commit();
            notify()->success('Presentation Successfully Created.');
            return redirect()->route('admin.presentations.create');
        } catch (\Exception $e) {
            DB::rollback();
            notify()->error('Something went wrong!'.$e->getMessage());
            return redirect()->route('admin.presentations.create');
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Presentation  $presentation
     * @return \Illuminate\Http\Response
     */
    public function show(Presentation $presentation)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Presentation  $presentation
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $presentation = Presentation::find($id);
        return view('backend.presentations.form', compact('presentation'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Presentation  $presentation
     * @return \Illuminate\Http\Response
     */
    public function update(UpdatePresentationRequest $request, Presentation $presentation)
    {
        DB::beginTransaction();
        try {
            $presentation->update([
                'title' => $request->title,
                'title_bn' => $request->title_bn,
                'details' => $request->details,
                'details_bn' => $request->details_bn,
                'is_active' => $request->is_active ? 1 : 0,
            ]);

            if($request->hasFile('image')) {
                $imageName = $this->uploadImageUpdate('images/presentations/'.$presentation->uuid.'/', $request->image, Str::slug($request->title), $presentation->image);
                Presentation::where('uuid', $presentation->uuid)
                            ->update(['image'=> $imageName]);
            }
            DB::commit();
            notify()->success('Presentation Successfully Updated.');
            return redirect()->route('admin.presentations.index');
        } catch (\Exception $e) {
            DB::rollback();
            notify()->error('Something went wrong!'.$e->getMessage());
            return redirect()->route('admin.presentations.index');
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Presentation  $presentation
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {

        DB::beginTransaction();
        try {
            $presentation = Presentation::find($id);
            $presentation->delete();

            DB::commit();
            notify()->info('Presentation Successfully Deleted.');
            return redirect()->route('admin.presentations.index');
        } catch (\Exception $e) {
            DB::rollback();
            notify()->error('Something went wrong!'.$e->getMessage());
            return redirect()->route('admin.presentations.index');
        }
    }
}
