<?php

namespace App\Http\Livewire\Backend;

use App\Events\AllowMessageEvent;
use App\Models\Message;
use Illuminate\Support\Carbon;
use Illuminate\Database\QueryException;
use Illuminate\Database\Eloquent\Builder;
use PowerComponents\LivewirePowerGrid\Button;
use PowerComponents\LivewirePowerGrid\Column;
use PowerComponents\LivewirePowerGrid\PowerGrid;
use PowerComponents\LivewirePowerGrid\PowerGridEloquent;
use PowerComponents\LivewirePowerGrid\PowerGridComponent;
use PowerComponents\LivewirePowerGrid\Traits\ActionButton;

class MessageTable extends PowerGridComponent
{
    use ActionButton;
    public string $primaryKey = 'messages.id';
    public string $sortField = 'messages.id';
    /*
    |--------------------------------------------------------------------------
    |  Features Setup
    |--------------------------------------------------------------------------
    | Setup Table's general features
    |
    */
    public function setUp()
    {
        $this->showPerPage()
            ->showExportOption('download', ['excel', 'csv'])
            ->showSearchInput();
    }

    /*
    |--------------------------------------------------------------------------
    |  Datasource
    |--------------------------------------------------------------------------
    | Provides data to your Table using a Model or Collection
    |
    */
    public function datasource(): ?Builder
    {
        return Message::query()
            ->join('users', function ($users) {
                $users->on('messages.user_id', '=', 'users.id');
            })
            ->orderBy('messages.id','desc')
            ->select([
            'messages.id',
            'messages.message',
            'messages.is_show',
            'users.name as user_name',
            'users.shareholder_name as shareholder_name',
            'users.company_name as company_name',
            'users.bo_id as bo_id',
            'users.user_type as user_type',
        ]);
    }

    /*
    |--------------------------------------------------------------------------
    |  Relationship Search
    |--------------------------------------------------------------------------
    | Configure here relationships to be used by the Search and Table Filters.
    |
    */
    public function relationSearch(): array
    {
        return [];
    }

    /*
    |--------------------------------------------------------------------------
    |  Add Column
    |--------------------------------------------------------------------------
    | Make Datasource fields available to be used as columns.
    | You can pass a closure to transform/modify the data.
    |
    */
    public function addColumns(): ?PowerGridEloquent
    {
        return PowerGrid::eloquent()
            ->addColumn('id')
            ->addColumn('bo_id')
            ->addColumn('shareholder_name')
            ->addColumn('company_name')
            ->addColumn('message')
            ->addColumn('is_show');

    }

    /*
    |--------------------------------------------------------------------------
    |  Include Columns
    |--------------------------------------------------------------------------
    | Include the columns added columns, making them visible on the Table.
    | Each column can be configured with properties, filters, actions...
    |
    */
    public function columns(): array
    {
        $canEdit = 1;
        return [
            Column::add()
                ->title(__('ID'))
                ->field('id'),

            Column::add()
                ->title(__('BO/Folio ID'))
                ->field('bo_id')
                ->searchable(),

            Column::add()
                ->title(__('User'))
                ->field('shareholder_name')
                ->searchable(),

            Column::add()
                ->title(__('Company'))
                ->field('company_name')
                ->searchable(),

            Column::add()
                ->title(__('Message'))
                ->field('message')
                ->searchable(),

            Column::add()
                ->title('Accept')
                ->field('is_show')
                ->toggleable($canEdit, 'yes', 'no'),
        ];
    }

    /*
    |--------------------------------------------------------------------------
    | Actions Method
    |--------------------------------------------------------------------------
    | Enable this section only when you have defined routes for these actions.
    |
    */


//    public function actions(): array
//    {
//       return [
//           Button::add('edit')
//               ->caption(__('Edit'))
//               ->class('bg-indigo-500 text-white')
//               ->route('message.edit', ['message' => 'id']),
//
//           Button::add('destroy')
//               ->caption(__('Delete'))
//               ->class('bg-red-500 text-white')
//               ->route('message.destroy', ['message' => 'id'])
//               ->method('delete')
//        ];
//    }


    /*
    |--------------------------------------------------------------------------
    | Edit Method
    |--------------------------------------------------------------------------
    | Enable this section to use editOnClick() or toggleable() methods.
    | Data must be validated and treated (see "Update Data" in PowerGrid doc).
    |
    */


    public function update(array $data ): bool
    {
       try {
           $data_message = Message::query()->find($data['id']);
           $data_message->is_show = $data_message->is_show == 1 ? 0 : 1;
           $data_message->save();
           $updated = true;
           broadcast(New AllowMessageEvent());
       } catch (QueryException $exception) {
           $updated = false;
       }

       return $updated;
    }

    public function updateMessages(string $status, string $field = '_default_message'): string
    {
        $updateMessages = [
            'success'   => [
                '_default_message' => __('Data has been updated successfully!'),
                //'custom_field' => __('Custom Field updated successfully!'),
            ],
            'error' => [
                '_default_message' => __('Error updating the data.'),
                //'custom_field' => __('Error updating custom field.'),
            ]
        ];

        return ($updateMessages[$status][$field] ?? $updateMessages[$status]['_default_message']);
    }


    public function template(): ?string
    {
        return null;
    }
}
