<?php

namespace App\Http\Controllers\Backend\Question;

use App\Http\Controllers\Controller;
use App\Http\Requests\Question\StoreQuestionRequest;
use App\Http\Requests\Question\UpdateQuestionRequest;
use App\Models\Question;
use App\Models\QuestionOptions;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Yajra\DataTables\DataTables;

class QuestionController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        return view('backend.questions.index');
    }

    public function datatableQuestion()
    {
        $search = \request('search');
        $questions = Question::orderBy('created_at','desc')
                            ->when($search, function ($q) use ($search){
                                $q->where('title','like','%'.$search.'%')
                                    ->orWhere('title_bn','like','%'.$search.'%')
                                    ->orWhere('details','like','%'.$search.'%')
                                    ->orWhere('details_bn','like','%'.$search.'%');
                            })->get();

        return DataTables::of($questions)
            ->addColumn('details', function ($question){
                return  wordwrap($question->details,70,"<br>\n");
            })->addColumn('details_bn', function ($question){
                return wordwrap($question->details_bn,100,"<br>\n");
            })
            ->addColumn('status', function($question){
                $str ='';
                $str = $str.'<div class="form-check form-switch">
									<input class="form-check-input" onclick="messageShow('.$question->id.')" type="checkbox" id="message-show-'.$question->id  .'"'. ($question->status == 1 ? "checked" : "") .'>
									<label class="form-check-label" for="message-show-'.$question->id  .'">End</label>
								</div>';
                return $str;
            })
            ->addColumn('action', function($question){
                $str ='';
                $str =$str.'<a href="'.route("admin.question.options",$question->id).'"  class="btn btn-outline-success me-2 "><i class="bx bx-plus-circle me-0"></i></a>';
                $str =$str.'<a href="'.route("admin.questions.edit",$question->uuid).'"  class="btn btn-outline-info me-3"><i class="bx bx-pencil me-0"></i></a>';
                $str = $str = $str.'<a class="btn btn-outline-danger" onclick="deleteData('.$question->id.')"><i class="bx bx-trash me-0"></i></a>
                        <form id="delete-form-'.$question->id.'" action="'. route('admin.questions.destroy',$question) .'" method="post"
                                      style="display: none;"> '.csrf_field().''.method_field('DELETE').' </form>';;
                return $str;
            })->addColumn('active', function($question){
                if($question->is_active == 1) {
                    $str = '<button type="button" class="btn btn-outline-success">Active</button>';
                }
                if($question->is_active == 0) {
                    $str = '<button type="button" class="btn btn-outline-danger">Inactive</button>';
                }

            return $str;
            })->rawColumns(['action', 'active', 'details', 'details_bn', 'status'])->make(true);
    }
    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('backend.questions.form');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(StoreQuestionRequest $request)
    {
        $option = array(
            $request->option_one,
            $request->option_two
            );

        DB::beginTransaction();
        try {
           $id =  Question::create([
                'title' => $request->title,
                'title_bn' => $request->title_bn,
                'details' => $request->details,
                'details_bn' => $request->details_bn,
                'proposed_resolution' => $request->proposed_resolution,
                'proposed_resolution_bn' => $request->proposed_resolution_bn,
                'total_shareholder_quantity' => $request->total_shareholder_quantity,
                'total_share_quantity' => $request->total_share_quantity,
                'is_active' => $request->is_active ? 1 : 0,
            ])->id;
            $data = [];
            $now = Carbon::now()->toDateTimeString();
            foreach ($option as $key => $value) {

                $data[] = array(
                    'option' => $value,
                    'question_id' => $id,
                    'type' => 'Yes/No',
                    'is_positive' => $key == 0 ? 1 : 0,
                    'uuid' => Str::uuid()->toString(),
                    'created_at' =>$now,
                    'updated_at' =>$now,
                );
            }
            QuestionOptions::insert($data);

            DB::commit();
            notify()->success('Agenda Successfully Created.');
            return redirect()->route('admin.questions.create');
        } catch (\Exception $e) {
            DB::rollback();
            notify()->error('Something went wrong!'.$e->getMessage());
            return redirect()->route('admin.questions.create');
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Question  $question
     * @return \Illuminate\Http\Response
     */
    public function show(Question $question)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Question  $question
     * @return \Illuminate\Http\Response
     */
    public function edit(Question $question)
    {
        $question->load('options');
        return view('backend.questions.form', compact('question'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Question  $question
     * @return \Illuminate\Http\Response
     */
    public function update(UpdateQuestionRequest $request, Question $question)
    {
        DB::beginTransaction();
        try {
            $question->update([
                'title' => $request->title,
                'title_bn' => $request->title_bn,
                'details' => $request->details,
                'details_bn' => $request->details_bn,
                'proposed_resolution' => $request->proposed_resolution,
                'proposed_resolution_bn' => $request->proposed_resolution_bn,
                'total_shareholder_quantity' => $request->total_shareholder_quantity,
                'total_share_quantity' => $request->total_share_quantity,
                'is_active' => $request->is_active ? 1 : 0,
            ]);

            DB::commit();
            notify()->success('Agenda Successfully Updated.');
            return redirect()->route('admin.questions.index');
        } catch (\Exception $e) {
            DB::rollback();
            notify()->error('Something went wrong!'.$e->getMessage());
            return redirect()->route('admin.questions.index');
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Question  $question
     * @return \Illuminate\Http\Response
     */
    public function destroy(Question $question)
    {
        DB::beginTransaction();
        try {
            $question->delete();
            DB::commit();
            notify()->info('Agenda Successfully Deleted.');
            return redirect()->route('admin.questions.index');
        } catch (\Exception $e) {
            DB::rollback();
            notify()->error('Something went wrong!'.$e->getMessage());
            return redirect()->route('admin.questions.index');
        }
    }

    public function questionVoteApprove($id)
    {
        $question  = Question::where('id', $id)->firstOrFail();
        $question->status = $question->status == 1 ? 0 : 1;
        $question->save();
        return response()->json($question);
    }
}
